--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

Simulator =
{
	lastSave = 0,
	travelCost = 487,
}

-- Some utilities
require("sim/simutils.lua")

-- Some dependencies
require("sim/quest_functions.lua")
require("sim/character_actions.lua")

-- Order may be significant for these includes
require("sim/item.lua")
require("sim/producttype.lua")
require("sim/character.lua")
require("sim/quest.lua")
require("sim/building.lua")
require("sim/factory.lua")
require("sim/laboratory.lua")
require("sim/market.lua")
require("sim/shop.lua")
require("sim/speakeasy.lua")
require("sim/port.lua")
require("sim/tips.lua")

require("sim/travel.lua")
require("sim/medals.lua")

-------------------------------------------------------------------------------

function Simulator:new(t)
	t = t or {} setmetatable(t, self) self.__index = self
	return t
end

-------------------------------------------------------------------------------
-- Initialization should be called once on startup

function Simulator:InitQuests()
	LQuest:ClearQuests()

	-- Load and post-process quest data separately
	dofile("quests/quests.lua")

--[[
	-- This was here for downloadable quest support... removing it now because
	-- we're not supporting this in the end...
	local qfiles = bsgLoadQuests()
	if type(qfiles) == "string" then
		qfiles = "return "..qfiles
		qfiles = loadstring(qfiles)
		if type(qfiles) == "function" then qfiles = qfiles() end
		if type(qfiles) == "table" then
			for _,f in pairs(qfiles) do
				if type(f) == "string" then
					DebugOut("LOADING QUESTS FROM:"..f)
					dofile(f)
				end
			end
		end
	end
]]--

	LQuest:PostProcessAll()
end

function Simulator:Initialize()
	-- Simulator data
	dofile("item/itemdefs.lua")
	dofile("ports/portlist.lua")
	
	-- Post-Process data as necessary
	LItem:PostProcessAll()
	LPort:PostProcessAll()
	LBuilding:PostProcessAll()
	
	self:InitQuests()
end

-------------------------------------------------------------------------------
-- Simulator reset

function Simulator:Reset(name)
	gSim = Simulator:new
	{
		player = name,		-- For string replacement
		name = name,
		mode = nil,
		rank = 0,
		money = 0,
		weeks = 0,
		days = 0,
		port = nil,
		quest = nil,
		questWeek = 0,
		hintWeek = 0,
		requireWeek = 0,
		gamble = 0,
		tip = 0,
		disaster = 0,
		firstcolor = false,
		stall = true,
		stallwarn = false,
		stallweeks = 0,
		
		messageTop = 1,
		messageQueue = false,
		inventoryTop = 1,
		messages = {"#", GetString("game_welcome") },
		
		medals = {},
		tips = {},
		travelPrices = nil,
		inventCount = 0,
		
		warnOnMiss = true,
		
		logo = {
			b="bg4",
			l="logo/logo10",lx=500,ly=500,la=1000,ls=1000,lf=0,lr=1000,
			t=name or "",to=5,ts=40,ta=1000,tch=500,tcs=500,tcv=500,tx=500,ty=500,tfo=1,
		}
	}

	-- Prepare messages
	gSim.messageText = table.concat(gSim.messages, "\n")
	
	-- Reset all game data
	LItem:ResetAll()
	LProductType:ResetAll()
	LPort:ResetAll()
	LBuilding:ResetAll()
	LCharacter:ResetAll()
	LQuest:ResetAll()
	LTip:ResetAll()

	LFactory:ProjectProduction()
end

-------------------------------------------------------------------------------
-- New Game

function Simulator:NewStoryMode()
	SetCurrentGameMode(0)

	self.mode = "story"
	self:SetPort("sanfrancisco")
	self:PrepareTravelPrices()
	
	local i = LItem:ByName("b_01")
	if i then
		i:EnableRecipe()
		i.newrecipe = false
	end
	
	-- Own the Sydney factory
	local b = LBuilding:ByName("sf_factory")
	b:MarkOwned()
	LFactory:ProjectProduction()
end

function Simulator:NewFreeMode()
	SetCurrentGameMode(1)

	self.mode = "free"
	self:SetPort("sanfrancisco")
	self:PrepareTravelPrices()
	
	-- Start with 20k
	self.money = 20000
	
	-- Default to NY, San Jose available
	local p = LPort:ByName("newyork")
	if p then p.available = true end
	local p = LPort:ByName("sanjose")
	if p then p.available = true end
	
	-- Enable all non-lab recipes
	for p in LItem:AllProducts() do
		if not p.lab then p:EnableRecipe() end
	end
	
	-- Own the SF factory
	local b = LBuilding:ByName("sf_factory")
	b:MarkOwned()
	LFactory:ProjectProduction()
end

-------------------------------------------------------------------------------
-- Save/Load

local function AppendToString(t, tString)
	for k,v in pairs(t) do
		-- If key is a number, just leave it off
		local key
		if type(k) == "number" then key = ""
		else key = k.."="
		end
	
		if type(v) == "string" then
			table.insert(tString, string.format("%s%q,", key, v))
		elseif type(v) == "number" or type(v) == "boolean" then
			table.insert(tString, string.format("%s%s,", key, tostring(v)))
		elseif type(v) == "table" then
			table.insert(tString, string.format("%s{", key))
			AppendToString(v, tString)
			table.insert(tString, "},")
		end
	end
end

function Simulator:BuildSaveTable()
	local t =
	{
		mode = self.mode,
		rank = self.rank,
		money = self.money,
		weeks = self.weeks,
		days = self.days,
		port = self.port.name,
		messages = self.messages,
		messageTop = self.messageTop,
		logo = self.logo,
		medals = self.medals,
		gamble = self.gamble,
		tip = self.tip,
		disaster = self.disaster,
		firstcolor = self.firstcolor,
		stall = self.stall,
		stallwarn = self.stallwarn,
		stallweeks = self.stallweeks,
		travelPrices = self.travelPrices,
		inventCount = self.inventCount,
	}
	
	if self.quest then
		t.quest = self.quest.name
		t.hintWeek = self.hintWeek
		t.requireWeek = self.requireWeek
	end
	t.questWeek = self.questWeek
	
	t.items = LItem:BuildSaveTable()
	t.ports = LPort:BuildSaveTable()
	t.buildings = LBuilding:BuildSaveTable()
	t.quests = LQuest:BuildSaveTable()
	t.chars = LCharacter:BuildSaveTable()
	t.tips = LTip:BuildSaveTable()
	
	return t
end

function Simulator:LoadSaveTable(t)
	self.mode = t.mode or "story"
	self.rank = t.rank or 0
	self.money = t.money or 0
	self.weeks = t.weeks or 1
	self.days = t.days or 0
	self.port = LPort:ByName(t.port) or LPort:ByName("sanfrancisco")
	self.quest = t.quest
	self.questWeek = t.questWeek or 0
	self.hintWeek = t.hintWeek or 0
	self.requireWeek = t.requireWeek or 0
	self.messages = t.messages or {}
	self.messageTop = t.messageTop or 1
	self.messageText = table.concat(self.messages, "\n")
	self.logo = t.logo or self.logo
	self.medals = t.medals or {}
	self.gamble = t.gamble or 0
	self.tip = t.tip or 0
	self.disaster = t.disaster or 0
	self.firstcolor = t.firstcolor or false
	self.stall = t.stall or true
	self.stallwarn = t.stallwarn or false
	self.stallweeks = t.stallweeks or 0
	
	self.inventCount = t.inventCount or 0
	
	self:PrepareTravelPrices(t.travelPrices)
	
	if self.quest ~= nil then
		self.quest = LQuest:ByName(self.quest)
		if self.quest then
			-- HACK: Grabbed from quest activation
			-- Auto-detect hint strings for this quest
			if not self.quest.hint then
				local key = self.quest.name .. "_hint"
				local hint = GetString(key)
				if hint ~= key then self.quest.hint = key end
			end
		end
	end
	
	self:GetScore()
	
	-- TODO: backwards compatibility during dev only: make sure logo names specify directory
	if t.logo then
		if t.logo.l and not string.find(t.logo.l, "/") then
			t.logo.l = "logo/"..t.logo.l
		end
	end

	LItem:LoadSaveTable(t.items)
	LPort:LoadSaveTable(t.ports)
	LBuilding:LoadSaveTable(t.buildings)
	LQuest:LoadSaveTable(t.quests)
	LCharacter:LoadSaveTable(t.chars)
	LTip:LoadSaveTable(t.tips)
	LFactory:ProjectProduction()
end

function Simulator:BuildSaveString()
	local t = self:BuildSaveTable()
	local tString = { "return {", }
	AppendToString(t, tString)
	table.insert(tString, "}")
	return table.concat(tString)
end

function Simulator:SaveGame()
	local s = self:BuildSaveString()
	bsgSaveGameString(s)
--	bsgStringToClipboard(s)
	Simulator.lastSave = bsgCurrentTime()
	
	-- Garbage collect on save games -- this will also kick in on the AutoSave timer, guaranteeing periodic garbage collection at a safe time
	bsgLuaGarbageCollect()
end

function Simulator:AutoSave()
	-- Auto save every 4 minutes
	local elapsedTime = bsgCurrentTime() - self.lastSave
	local remain = (4 * 60 * 1000) - elapsedTime
	if remain <= 0 then self:SaveGame() end
end

function Simulator:LoadGame()
	local name = GetUserName(GetCurrentUser())
	Simulator:Reset(name)
	
	local mode = GetCurrentGameMode()
	local ok = false
	
	-- Force load story table regardless of requested mode
	SetCurrentGameMode(0)
	local tStory = nil
	local s = bsgLoadGameString()
	if s ~= "" then
		tStory = loadstring(s)
		if type(tStory) == "function" then tStory = tStory() end
	end
	
	if mode == 0 then
		-- Looking for Story mode, that's it...
		ok = true
		if type(tStory) == "table" then
			gSim:LoadSaveTable(tStory)
		else
			gSim:NewStoryMode()
			gSim:SaveGame()
		end
	elseif mode == 1 and type(tStory) == "table" then
		-- Load or prepare Free Play mode table
		ok = true
		SetCurrentGameMode(1)
		local tFree = nil
		local s = bsgLoadGameString()
		if s ~= "" then
			tFree = loadstring(s)
			if type(tFree) == "function" then tFree = tFree() end
		end
		
		if type(tFree) == "table" then
			gSim:LoadSaveTable(tFree)
			gSim.mode = "free"
		else
			gSim:NewFreeMode()
			gSim:SaveGame()
		end
		
		-- Adjust selected Free Play settings according to Story Mode settings:
		-- Enable ports open in Story mode
		for name,data in pairs(tStory.ports) do
			local p = LPort:ByName(name)
			if p then p.available = true end
		end
		
		-- Enable extra recipes discovered in Story mode
		for name,data in pairs(tStory.items) do
			local item = LItem:ByName(name)
			if item and data.known then
				-- Known counts will be updated below
				item.known = true
			end
		end
		
		-- Prepare medals according to Story Mode
		gSim.medals = tStory.medals
		gSim.rank = tStory.rank
	end
	
	if ok then
		Simulator.lastSave = bsgCurrentTime()
		LProductType:CountAllKnownRecipes()
		ResetLedger(gSim.money or 0)
	else
		gSim = nil
	end
	
	return ok
end

-------------------------------------------------------------------------------
-- High Scores

-- TODO: figure out a better score system!

function Simulator:GetScore()
	-- TODO: Freeze score at top level?

	local weeks = self.weeks or 1
	if weeks > 999999999 then weeks = 999999999 end
	if weeks <= 0 then weeks = 1 end

	local money = self.money or 0
	if money > 4294967295 then money = 4294967295 end
	if money <= 0 then money = 0 end

	local score = money
	score = bsgFloor(money / weeks + 0.5)
	if score > 999999999 then score = 999999999 end
	self.score = score
	
	if gSim.rank == 0 then score = 0 end

	return score,weeks,money
end

function Simulator:GetMedalFlags()
	local flag = 1
	local all = 0
	for i,m in ipairs(Simulator._Medals) do
		if self.medals[m.name] then all = all + flag end
		flag = flag * 2
	end
	return all
end

function Simulator:LogScore()
	local score, weeks, money = self:GetScore()
	local rank = self.rank or 0
	if score > 0 then
		-- Prepare server data string: R-WWWWWWWWW-MMMMMMMMM
		local stringData = string.format("%d-%09d-%09d", rank, weeks, money)
		bsgLogScore(score, stringData, self:GetMedalFlags())
	end
end

-------------------------------------------------------------------------------
-- UI Help

function Simulator:RolloverCurrentQuest()
	if self.quest then
		return self.quest:Rollover()
	end
end

function Simulator:QueueMessage(message)
	self.messageQueue = true
	table.insert(self.messages, message)
	-- Keep 30 messages, plus the leading "#"
	if (table.getn(self.messages) > 31) then table.remove(self.messages,2) end
end

function Simulator:ConcatenateMessages()
	if self.messageQueue then
		self.messageQueue = false
		self.messageText = table.concat(self.messages, "\n")
		SetLabel("messageText", self.messageText)
--		local newTop = bsgGetMaxScroll("messageText")
		local newTop = bsgGetMaxScroll("messageText") + 1
		
--		if newTop == 0 then newTop = 1
--		elseif newTop == -1 then newTop = self.messageTop
--		end
		if newTop == 0 then newTop = self.messageTop end

		self.messageTop = newTop
	end
end

function Simulator:FlushMessages()
	if self.messageQueue then
		self:ConcatenateMessages()
		if UpdateLedgerContents then UpdateLedgerContents("messages") end
	end
end

function Simulator:Message(message)
	self:QueueMessage(message)
	self:FlushMessages()
end

-------------------------------------------------------------------------------
-- Money/Inventory adjustment

function Simulator:AdjustMoney(n, reason)
	-- Maximum a 32-bit unsigned int can hold...
	if (n > 4294967295 - self.money) then self.money = 4294967295
	else self.money = self.money + n
	end
	if self.money < 0 then self.money = 0 end
	
	if reason then
		if n < 0 then n = -n end
		reason = GetString(reason, bsgDollars(n))
		self:QueueMesssage(reason)
	end
	
	self:GetScore()
	UpdateBadge(self)
end

function Simulator:InventoryChanged()
	LFactory:ProjectProduction()
	UpdateLedgerContents("inventory")
	UpdateLedgerContents("factories")
end

-------------------------------------------------------------------------------
-- Lose condition
-- Failure state: < gSim.travelCost, no products in inventory OR no shop in town

function Simulator:CheckLoseCondition()
	local keepPlaying = true
	if self.weeks > 0 and self.money < gSim.travelCost and not gTravelActive then
	
		local bad = true
		
		-- If there's a shop in town, there's a chance to sell something
		if self.port.hasShop then
			for item in LItem:AllProducts() do
				if item.inventory > 0 then
					bad = false
					break
				end
			end

			-- If nothing in inventory, there's a chance a factory might produce something
			if bad then
				self:Tick()
				for item in LItem:AllProducts() do
					if item.inventory > 0 then
						bad = false
						break
					end
				end

			end
		end
	
		-- Still bad -- if player is not on a quest, offer one of the "nomoney" quests
		if bad and not gSim.quest then
			local q = LQuest:ByName("nomoney")
			if q and q.complete then q = LQuest:ByName("nomoney2") end
			if q and q.complete then q = LQuest:ByName("nomoney3") end
			if q and q.complete then q = LQuest:ByName("nomoney4") end
			if q then
				q:Offer()
				if self.money >= gSim.travelCost then
					bad = false
					-- FIRSTPEEK: Loan, <stamp>, weeks, current-money
					if fpWrite then fpWrite { "Loan", gSim.weeks, gSim.money } end
				end
			end
		end
		
		-- Still bad?? -- have money lender offer to liquidate products...
		local productsOwned = false
		if bad then
			local cash = 0
			for prod in LItem:OwnedProducts() do
				cash = prod.inventory * prod.lowcost
			end
			
			if cash > 0 then
				productsOwned = true
				local character = LCharacter:ByName("lender")
				local liquidate = "#"..GetString("product_liquidate", bsgDollars(cash))
				local yn = DisplayDialog { "ui/chartalk.lua", char=character, body=liquidate, yes="yes", no="no" }
				if yn == "yes" then
					for prod in LItem:OwnedProducts() do
						cash = prod.inventory * prod.lowest
						gSim:QueueMessage(GetString("liquidate_product", tostring(prod.inventory), GetString(prod.name), bsgDollars(prod.lowest)))
						prod.inventory = 0
						gSim:AdjustMoney(cash)
					end
					self:InventoryChanged()
					productsOwned = false
					if self.money >= gSim.travelCost then bad = false end
					-- FIRSTPEEK: LiquidateProduct, <stamp>, weeks, current-money
					if fpWrite then fpWrite { "LiquidateProduct", gSim.weeks, gSim.money } end
				end
			end
		end

		-- Still bad?? -- have money lender offer to liquidate ingredients...
		local ingredientsOwned = false
		if bad then
			local cash = 0
			for item in LItem:OwnedIngredients() do
				cash = item.inventory * item.low
			end
			
			if cash > 0 then
				ingredientsOwned = true
				local character = LCharacter:ByName("lender")
				local liquidate = "#"..GetString("ingredient_liquidate", bsgDollars(cash))
				local yn = DisplayDialog { "ui/chartalk.lua", char=character, body=liquidate, yes="yes", no="no" }
				if yn == "yes" then
					for item in LItem:OwnedIngredients() do
						cash = item.inventory * item.low
						gSim:QueueMessage(GetString("liquidate_ingredient", tostring(item.inventory), GetString(item.name), bsgDollars(item.low)))
						item.inventory = 0
						gSim:AdjustMoney(cash)
					end
					self:InventoryChanged()
					ingredientsOwned = false
					if self.money >= gSim.travelCost then bad = false end
					-- FIRSTPEEK: LiquidateIngredients, <stamp>, weeks, current-money
					if fpWrite then fpWrite { "LiquidateIngredients", gSim.weeks, gSim.money } end
				end
			end
		end
		
		-- STILL bad? If the player is not just holding out on liquidating, give them some cash...
		if bad and (not productsOwned) and (not ingredientsOwned) then
			local cash = gSim.travelCost * 2
			local character = LCharacter:ByName("lender")
			local finished = "#"..GetString("game_over", bsgDollars(cash))
			local yn = DisplayDialog { "ui/chartalk.lua", char=character, body=finished, yes="yes", no="no" }
			if yn == "yes" then
				gSim:AdjustMoney(cash, angel)
				-- FIRSTPEEK: FreeCash, <stamp>, weeks, current-money
				if fpWrite then fpWrite { "FreeCash", gSim.weeks, gSim.money } end
				gSim:Tick()
			else
				keepPlaying = false
				-- FIRSTPEEK: GameOver, <stamp>, weeks, current-money
				if fpWrite then fpWrite { "GameOver", gSim.weeks, gSim.money } end
				DisplayDialog { "ui/chartalk.lua", char=character, body="angel_no" }
				SwapToModal("ui/mainmenu.lua")
			end
		end
	end
	
	return keepPlaying
end

-------------------------------------------------------------------------------
-- Travel pricing

function Simulator:PrepareTravelPrices(t)
	self.travelPrices = t or {}
	if t then
		-- Restore travel prices from the table
		for p in LPort:AllPorts() do
			local route = self.port:GetRoute(p.name)
			if route and t[p.name] then route.price = t[p.name] end
		end
	else
		-- Prepare travel prices and store them
		for p in LPort:AllPorts() do
			local route = self.port:GetRoute(p.name)
			if route then
				-- Adjust travel prices between 90 and 100% of full price
				price = route.totalWeeks * gSim.travelCost
				
				-- Travel in or out of secret ports is more expensive (2x) for the 2 weeks
				-- it takes to get from any of them to anywhere else
				if gSim.port.hidden or p.hidden then price = price + 2 * gSim.travelCost end
				
				price = bsgFloor(price * bsgRandom(900,1000) / 1000 + 0.5)
				route.price = price
				self.travelPrices[p.name] = price
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Tick

function Simulator:Expiration()
	local change = false
	for item in LItem:AllItems() do
		-- After 32 weeks of non-use, gradually expire inventory at 20% per week
		local d = self.weeks - item.usetime
		if item.inventory > 0 and d > 32 then
			-- Don't expire Truffles or Exotics
			if not (item.recipe and (item.type.name == "truffle" or item.type.name == "exotic")) then
				local newCount = bsgFloor(item.inventory * .80)
				item.inventory = newCount
				change = true
				
				-- This is part of a Tick, messages will be flushed at end of Tick
				self:QueueMessage(GetString("item_expire", GetString(item.name)))
			end
		end
	end
	if change then self:InventoryChanged() end
end

function Simulator:FirstPeekProgress()
	if fpWrite then
		local totalSold = 0
		local totalInventory = 0
		local squaresSold, infusionsSold, saucesSold = 0,0,0
		for item in LItem:AllProducts() do
			totalSold = totalSold + item.sold
			totalInventory = totalInventory + item.inventory
			
			if item.type.name == "square" then squaresSold = squaresSold + item.sold
			elseif item.type.name == "infusion" then infusionsSold = infusionsSold + item.sold
			elseif item.type.name == "sauce" then saucesSold = saucesSold + item.sold
			end
		end
	
		-- FIRSTPEEK: Progress, <stamp>, weeks, money, factories, totalSold, totalInventory, squaresKnown, squaresSold, infusionsKnown, infusionsSold, saucesKnown, saucesSold, outputSanFrancisco, outputNewYork, outputParis, outputMahajanga, stallWeeks
		fpWrite { "Progress", gSim.weeks, gSim.money, gSim.factoriesOwned, totalSold, totalInventory,
			LProductType:ByName("square").knownCount,
			squaresSold,
			LProductType:ByName("infusion").knownCount,
			infusionsSold,
			LProductType:ByName("sauce").knownCount,
			saucesSold,
			LFactory:ByPort("sanfrancisco").rate,
			LFactory:ByPort("newyork").rate,
			LFactory:ByPort("paris").rate,
			LFactory:ByPort("mahajanga").rate,
			gSim.stallweeks,
		}
	end
end

function Simulator:Tick()
	LFactory:ProductionRun()
	LCharacter:MoodTick()
	self:Expiration()
	
	-- Prices change weekly -- (TODO: when not travelling)
	LItem:PreparePrices()
	LTip:ApplyAll()

	self.days = 0
	self.weeks = self.weeks + 1
	self:GetScore()
	UpdateBadge(self)
	
--	self:QueueMessage(GetString("tick"))
	self:QueueMessage(bsgDate(gSim.weeks, gSim.days))
	self:FlushMessages()
	
	if fpWrite and bsgMod(gSim.weeks, 6) == 5 then self:FirstPeekProgress() end
end

function Simulator:TickDay()
	self.days = self.days + 1
	if self.days >= 4 then self:Tick() end
	UpdateBadge(self)
end
